<?php
// 变量获取 - 放在文件开头
$url        = '';
$defaultIco = 'default/' . mt_rand( 1, 11 ) . '.png';
$expire     = 2592000; // 缓存有效期30天
$cacheDir   = 'cache'; // 缓存目录

// 防止其他网站使用API，替换为您的域名
$allowedDomains = array(
	'localhost',
	'127.0.0.1',
	'zmingcx.com',
);

// 防止其他网站使用API
if ( ! empty( $allowedDomains ) ) {
	$referer = $_SERVER['HTTP_REFERER'] ?? '';

	// 如果有Referer，检查是否来自允许的域名
	if ( ! empty( $referer ) ) {
		$parsed = parse_url( $referer );

		// 检查parse_url是否成功并且包含host
		if ( $parsed === false || ! isset( $parsed['host'] ) ) {
			http_response_code( 403 );
			header( 'Content-Type: application/json' );
			echo json_encode(
				array(
					'error'   => 'Access denied',
					'message' => 'Invalid referer',
				)
			);
			exit;
		}

		$refererHost = $parsed['host'];
		$isAllowed   = false;

		foreach ( $allowedDomains as $domain ) {
			if ( $refererHost === $domain || str_ends_with( $refererHost, '.' . $domain ) ) {
				$isAllowed = true;
				break;
			}
		}

		if ( ! $isAllowed ) {
			http_response_code( 403 );
			header( 'Content-Type: application/json' );
			echo json_encode(
				array(
					'error'   => 'Access denied',
					'message' => 'This API is not available for external use',
				)
			);
			exit;
		}
	}
	// 如果没有Referer（直接访问），允许通过
}

// 获取URL参数 - 支持PATH_INFO和GET参数两种方式
if ( isset( $_SERVER['PATH_INFO'] ) && ! empty( trim( $_SERVER['PATH_INFO'], '/' ) ) ) {
	$url = trim( $_SERVER['PATH_INFO'], '/' );
} elseif ( isset( $_GET['url'] ) && ! empty( trim( $_GET['url'] ) ) ) {
	$url = trim( $_GET['url'] );
}

// 验证URL参数
if ( empty( $url ) ) {
	http_response_code( 404 );
	exit( 'URL parameter is required' );
}

// URL格式基本验证 - 增强安全性
if ( ! filter_var( $url, FILTER_VALIDATE_URL ) && ! preg_match( '/^[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}/', $url ) ) {
	http_response_code( 400 );
	exit( 'Invalid URL format' );
}

// 检查依赖文件和目录
if ( ! file_exists( './favicon.php' ) ) {
	http_response_code( 500 );
	exit( 'Favicon processor not found' );
}

// 检查默认图标目录
$defaultDir = dirname( $defaultIco );
if ( ! is_dir( $defaultDir ) ) {
	http_response_code( 500 );
	exit( 'Default icons directory not found: ' . $defaultDir );
}

// 检查默认图标文件是否存在
if ( ! file_exists( $defaultIco ) ) {
	// 如果随机文件不存在，使用第一个可用的文件
	$availableIcons = glob( $defaultDir . '/*.png' );
	if ( empty( $availableIcons ) ) {
		http_response_code( 500 );
		exit( 'No default icons available' );
	}
	$defaultIco = $availableIcons[0];
}

// 检查缓存目录
if ( ! is_dir( $cacheDir ) ) {
	if ( ! mkdir( $cacheDir, 0755, true ) ) {
		error_log( '创建缓存目录失败：' . $cacheDir );
		http_response_code( 500 );
		exit( 'Cache directory creation failed' );
	}
}

// 检查缓存目录写权限
if ( ! is_writable( $cacheDir ) ) {
	http_response_code( 500 );
	exit( 'Cache directory is not writable' );
}

require './favicon.php';

// 初始化favicon处理类
$favicon = new \Beico\Favicon\Favicon();
$favicon->setDefaultIcon( $defaultIco );

// 格式化URL
$formatUrl = $favicon->formatUrl( $url );
if ( ! $formatUrl ) {
	http_response_code( 400 );
	exit( 'URL format processing failed' );
}

// PHP逻辑处理部分
if ( $expire == 0 ) {
	// 不使用缓存
	$favicon->getFavicon( $formatUrl, false );
	exit;
} else {
	// 使用缓存
	$defaultMD5 = md5( file_get_contents( $defaultIco ) );
	$data       = Cache::get( $formatUrl, $defaultMD5, $expire );

	if ( $data !== null ) {
		// 输出缓存的数据
		foreach ( $favicon->getHeader() as $header ) {
			@header( $header );
		}
		echo $data;
		exit;
	}

	// 获取新的favicon
	try {
		$content = $favicon->getFavicon( $formatUrl, true );
		if ( md5( $content ) == $defaultMD5 ) {
			$expire = 43200; // 如果返回默认图标，设置过期时间为12小时
		}

		// 设置缓存并输出
		Cache::set( $formatUrl, $content, $expire );
		foreach ( $favicon->getHeader() as $header ) {
			@header( $header );
		}
		echo $content;
		exit;
	} catch ( Exception $e ) {
		error_log( 'Favicon获取失败: ' . $e->getMessage() );
		http_response_code( 500 );
		exit( 'Favicon processing failed' );
	}
}

/**
 * 缓存处理类
 * 提供favicon缓存的读写功能
 */
class Cache {

	/**
	 * 获取缓存数据
	 *
	 * @param string $key 缓存键（URL）
	 * @param string $default 默认图标的MD5值
	 * @param int    $expire 过期时间（秒）
	 * @return string|null 缓存数据或null
	 */
	public static function get( $key, $default, $expire ) {
		$dir = 'cache'; // 图标缓存目录

		// 安全解析URL - 遵循parse_url安全使用规范
		$parsed_url = parse_url( $key );
		if ( $parsed_url === false ) {
			error_log( 'URL解析失败: ' . $key );
			return null;
		}

		// 检查host键是否存在
		if ( ! isset( $parsed_url['host'] ) ) {
			error_log( 'URL中缺少host部分: ' . $key );
			return null;
		}

		$f = $parsed_url['host'];
		$a = $dir . '/' . $f . '.txt';

		if ( is_file( $a ) ) {
			$data = file_get_contents( $a );
			if ( $data === false ) {
				error_log( '读取缓存文件失败: ' . $a );
				return null;
			}

			if ( md5( $data ) == $default ) {
				$expire = 43200; // 如果返回默认图标，过期时间为12小时
			}

			if ( ( time() - filemtime( $a ) ) > $expire ) {
				return null;
			} else {
				return $data;
			}
		} else {
			return null;
		}
	}

	/**
	 * 设置缓存数据
	 *
	 * @param string $key 缓存键（URL）
	 * @param string $value 要缓存的数据
	 * @param int    $expire 过期时间（秒）
	 * @return void
	 */
	public static function set( $key, $value, $expire ) {
		$dir = 'cache'; // 图标缓存目录

		// 安全解析URL - 遵循parse_url安全使用规范
		$parsed_url = parse_url( $key );
		if ( $parsed_url === false ) {
			error_log( 'URL解析失败: ' . $key );
			return;
		}

		// 检查host键是否存在
		if ( ! isset( $parsed_url['host'] ) ) {
			error_log( 'URL中缺少host部分: ' . $key );
			return;
		}

		$f = $parsed_url['host'];
		$a = $dir . '/' . $f . '.txt';

		// 如果缓存目录不存在则创建
		if ( ! is_dir( $dir ) ) {
			if ( ! mkdir( $dir, 0755, true ) ) {
				error_log( '创建缓存目录失败：' . $dir );
				return;
			}
		}

		// 检查是否需要更新缓存
		if ( ! is_file( $a ) || ( time() - filemtime( $a ) ) > $expire ) {
			$imgdata = fopen( $a, 'w' );
			if ( $imgdata === false ) {
				error_log( '无法打开缓存文件：' . $a );
				return;
			}

			if ( fwrite( $imgdata, $value ) === false ) {
				error_log( '写入缓存文件失败：' . $a );
			}

			fclose( $imgdata );
		}
	}
}
